/*= Instrument bp2318  ====================================================*/
/* LabWindows 2.2 Instrument Driver                                        */
/* Original Release: bp2318.c   V1.0                      Date: 13.06.94   */
/* By: M.Westermann                                                        */
/* Originally written in C                                                 */
/* Modification History: V1.1 Date 05.08.94 M.Westermann                   */
/*                       Software mistakes in 2318                         */
/* Modification History: V1.2 Date 08.06.98 LU/M.Westermann                */
/*                       Range store in EERPROM                            */
/* Modification History: V1.3 Date 01.08.02 T. Meder                       */
/*                       Measurement value 0.0 comes incorrect over RS     */
/*						 interface.										   */
/*							Release: V1.4  03.02.2003 T. Meder             */			   
/*                                         Changes: CloseCom added         */
/*						 		     V1.5  09.04.2003 T. Meder             */
/*										   Routine: bp2318_meas_read	   */
/*										   Check, if a measurement value   */
/*                                         is available                    */
/*						 		     V1.6  13.11.2006 T. Meder             */
/*                                         COM-Ports enhanced to 8 ports   */
/*                                   V1.8  14.08.2007 T. Meder             */
/*                                         Routine: bp2318_meas_read	   */
/*                                         CompareStrings replaced by      */
/*                                         FindPattern                     */
/*																	       */ 
/*=========================================================================*/

/* Remark : This instrument driver include the procedures for
            ANSI X3.28(A3,A4)  data communication control
            Uilitiy's : send_x3_28(),bp2318_receive_x3_28()               */

/*= INSTRUMENT-DEPENDENT COMMAND ARRAYS ===================================*/

#include <userint.h>
#include <rs232.h>
#include <formatio.h>
#include <utility.h>
#include <gpib.h>
#include <ansi_c.h>

#include "bp2318.h"

static char out_buf[MAX_BYTE];               /* buffer for send */
static char in_buf[MAX_BYTE];                /* buffer for receive */
static char swap_buf[MAX_BYTE];              /* global buffer */
static char buf[MAX_BYTE];                   /* Buffer fr functions*/
static int int_level[4] ;                    /* RS232-Interrupt-Level Array*/
static int dev_port[BP2318_MAX_INSTR];       /* Device Port*/
static int dev_group_addr[BP2318_MAX_INSTR]; /* Device Group Adress*/
static int dev_user_addr[BP2318_MAX_INSTR];  /* Device User Adress */
static char *far func_load[2];               /* SCPI-Command-Arrays*/
static char *far func_auto_zero[2];
static char *far func_range[10];
static char *far func_unit[10];
static char *far func_limit_20mV[4];
static char *far func_temp_comp[4];
static char *far func_comp_state[2];
static char *far func_keyboard[2];

static int setup_first ;                  /* remark the first initial*/
static double std_timeout;                /* remark the user timeout*/
static int max_bcc_repeat;                /* Max-Bcc-Repeat */
static int bcc_repeat;                    /* Bcc-Repeat counter*/
static int bcc_state;                     /* Blockcheck state ON or OFF*/
static int bp2318_err;                    /* Include the Errors*/

/*= UTILITY ROUTINES ======================================================*/
int _VI_FUNC  bp2318_open_instr (int,int,int,int);
int _VI_FUNC  bp2318_device_closed (int);
int _VI_FUNC  bp2318_close_instr (int);
int _VI_FUNC  bp2318_invalid_integer_range (int,int,int,int);
int _VI_FUNC  bp2318_invalid_real_range (double ,double,double,int);
int _VI_FUNC  bp2318_send_x3_28(int,char*,char*);        /* Send a comand and message
                                                  and wait for ACK or NAK*/
int _VI_FUNC  bp2318_receive_x3_28(int,char*,char*,char*);/* Receive a message*/

int _VI_FUNC  bp2318_receive_and_ack(int,char*);

int _VI_FUNC  bp2318_send_x3_28_message(int,char*,char*);/*Send a comand and a message*/
int _VI_FUNC  bp2318_calculate_bcc(char*);               /* calculate the Blockcheck*/
int _VI_FUNC  bp2318_open_instr (int,int ,int, int );    /* Search a free place in the*/
                                               /* Device arrays and open the*/
                                               /* COM-Port*/
void bp2318_setup_arrays (void);               /* refresh the function arrays*/
int _VI_FUNC  bp2318_read_range (int ,int* );

/*=========================================================================*/
/* This function opens a com port for the instrument module                */
/*=========================================================================*/
int _VI_FUNC  bp2318_init(int port,int baud_rate,double timeout,int set_bcc_state,
                int g_addr,int u_addr,int dev_reset,int *instrID)
{
    int ID;

    bp2318_err = 0;
    bp2318_setup_arrays ();

    if (bp2318_invalid_integer_range (port, 1,8, -1) != 0)
     return bp2318_err;
    if (bp2318_invalid_integer_range (baud_rate, 1200, 19200, -2) != 0)
     return bp2318_err;
    if (bp2318_invalid_real_range (timeout, 0.0F, 72000.0F, -3) != 0)
     return bp2318_err;
    if (bp2318_invalid_integer_range (set_bcc_state, 0,1, -4) != 0)
     return bp2318_err;
    if (bp2318_invalid_integer_range (g_addr, 0,15, -5) != 0)
     return bp2318_err;
    if (bp2318_invalid_integer_range (u_addr, 0,15, -6) != 0)
     return bp2318_err;
    if (bp2318_invalid_integer_range (dev_reset, 0,1, -7) != 0)
     return bp2318_err;

    ID = bp2318_open_instr (port,baud_rate,g_addr,u_addr);
    if(ID <= 0)
     return bp2318_err;

    SetComTime (port, timeout);
    std_timeout = timeout;

    if (rs232err != 0)
        return (rs232err);

    max_bcc_repeat = 5;       /* default*/
    bcc_state = set_bcc_state;

    if(dev_reset)
    {
     if(bp2318_send_scpi(ID,"*RST") != 0)
      return bp2318_err;
    }

    *instrID = ID;
    return(bp2318_err);
}
/*=========================================================================*/
/* Function: Close                                                         */
/* Purpose:  This function closes the instrument.                          */
/*=========================================================================*/
int _VI_FUNC  bp2318_close (int instrID)
{
    bp2318_err = 0;
    if (bp2318_invalid_integer_range (instrID, 1, BP2318_MAX_INSTR -1, -1) != 0)
        return bp2318_err;

    if (bp2318_device_closed (instrID))
        return bp2318_err;

    bp2318_close_instr (instrID);
    return bp2318_err;
}
/*=========================================================================*/
/* Function: Set Config                                                        */
/* Purpose:  This function set the configuration of the instrument.        */
/*=========================================================================*/
int _VI_FUNC  bp2318_set_config(int instrID,int load,int auto_zero,int range,
                      int limit_20mV,int temp_comp)
{
    bp2318_err = 0;
    if (bp2318_invalid_integer_range (instrID, 1, BP2318_MAX_INSTR -1, -1) != 0)
        return bp2318_err;
    if (bp2318_invalid_integer_range (load, 0, 1, -2) != 0)
        return bp2318_err;
    if (bp2318_invalid_integer_range (auto_zero, 0, 1, -3) != 0)
        return bp2318_err;
    if (bp2318_invalid_integer_range (range, 0, 8, -4) != 0)
        return bp2318_err;
    if (bp2318_invalid_integer_range (limit_20mV, 0, 3, -5) != 0)
        return bp2318_err;
    if (bp2318_invalid_integer_range (temp_comp, 0, 3, -6) != 0)
        return bp2318_err;

    if(bp2318_send_scpi(instrID,func_load[load]) != 0)
      return bp2318_err;
    if(bp2318_send_scpi(instrID,func_auto_zero[auto_zero]) != 0)
      return bp2318_err;
    if(!load)
    {
     if(bp2318_send_scpi(instrID,func_limit_20mV[limit_20mV]) != 0)
      return bp2318_err;
    }
    if(bp2318_send_scpi(instrID,func_range[range]) != 0)
      return bp2318_err;
	if(bp2318_send_scpi(instrID,"SENS:RES:RANG:STOR") != 0) /*demo2318 V1.2 LU*/
      return bp2318_err;
      
    if(bp2318_send_scpi(instrID,func_temp_comp[temp_comp]) != 0)
      return bp2318_err;

    return bp2318_err;
}
/*=========================================================================*/
/* Function: Get Config                                                        */
/* Purpose:  This function get the configuration of the instrument.        */
/*=========================================================================*/
int _VI_FUNC  bp2318_get_config(int instrID,int* load,int* auto_zero,int* range,
                      int* limit_20mV,int* temp_comp)
{
    int help;

    bp2318_err = 0;
    if (bp2318_invalid_integer_range (instrID, 1, BP2318_MAX_INSTR -1, -1) != 0)
        return bp2318_err;

    if(bp2318_receive_scpi(instrID,":SENS:RES:LOAD?",buf) != 0)
      return bp2318_err;

    if(CompareStrings (buf, 1, "REAL", 1, 1) == 0)
     *load = 0;
    else if(CompareStrings (buf, 1, "COMP", 1, 1) == 0)
     *load = 1;
    else
     bp2318_err = 304;

    if(bp2318_receive_scpi(instrID,":CAL:ZERO?",buf) != 0)
      return bp2318_err;
    Scan(buf,"%s>%i",auto_zero);



    if(bp2318_receive_scpi(instrID,":SENS:RES:RANG:AUTO?",buf) != 0)
      return bp2318_err;

    Scan(buf,"%s>%i",&help);
    if(help == 1)   /* Auto */
     *range = 0;
    else
    {
     if(bp2318_read_range(instrID,range) != 0)
      return bp2318_err;

    }
    if(bp2318_receive_scpi(instrID,":SOUR:VOLT:LIM?",buf) != 0)
      return bp2318_err;

    if(CompareStrings (buf, 1, "NONE", 1, 1) == 0)
      *limit_20mV = 0;
    else if(CompareStrings (buf, 1, "20MV,1A", 1, 1) == 0)
     *limit_20mV = 1;
    else if(CompareStrings (buf, 1, "20MV,100MA", 1, 1) == 0)
     *limit_20mV = 2;
    else if(CompareStrings (buf, 1, "20MV,10MA", 1, 1) == 0)
     *limit_20mV = 3;
    else
     bp2318_err = 304;

    if(bp2318_receive_scpi(instrID,":SENS:CORR:TCOM:MED?",buf) != 0)
      return bp2318_err;

    if(CompareStrings (buf, 1, "NONE", 1, 1) == 0)
      *temp_comp = 0;
    else if(CompareStrings (buf, 1, "CU/AL", 1, 1) == 0)
     *temp_comp = 1;
    else if(CompareStrings (buf, 1, "MS63", 1, 1) == 0)
     *temp_comp = 2;
    else if(CompareStrings (buf, 1, "MS80", 1, 1) == 0)
     *temp_comp = 3;
    else
     bp2318_err = 304;

    return bp2318_err;
}
/*=========================================================================*/
/* Function: Measure Start                                                 */
/* Purpose:  This function start the measring                              */
/*=========================================================================*/
int _VI_FUNC  bp2318_meas_start(int instrID)
{
    bp2318_err = 0;
    if (bp2318_invalid_integer_range (instrID, 1, BP2318_MAX_INSTR -1, -1) != 0)
        return bp2318_err;

    if(bp2318_send_scpi(instrID,"INIT") != 0)
      return bp2318_err;

    return bp2318_err;
}
/*=========================================================================*/
/* Function: Measure Stop                                                  */
/* Purpose:  This function stop the measring                               */
/*=========================================================================*/
int _VI_FUNC  bp2318_meas_read(int instrID,double *value,char * unit)
{
    int flag,range;
    int intRet;

    bp2318_err = 0;
    if (bp2318_invalid_integer_range (instrID, 1, BP2318_MAX_INSTR -1, -1) != 0)
        return bp2318_err;

    if(bp2318_receive_scpi(instrID,"STAT:OPER:COND?",buf) != 0)
      return bp2318_err;

    Scan(buf,"%s>%i",&flag);
    if(!(flag & BP2318_MEASURING))
    {
     bp2318_err = 302;    /* There is no measuring started */
     return bp2318_err;
    }

    if(bp2318_receive_scpi(instrID,"STAT:QUES:COND?",buf) != 0)
      return bp2318_err;
    
    Scan(buf,"%s>%i",&flag);
    
    if(flag & BP2318_OVERRANGE)
    {
     bp2318_err = 303;    /* Overrange*/
     return bp2318_err;
    }

/*
	// Nur fr 2316 aktivieren:
    // 09.04.2003 / MT: Check Status Byte -> Measvalue Available?
    if(bp2318_receive_scpi(instrID,"*STB?",buf) != 0)
      return bp2318_err;

    Scan(buf,"%s>%i",&flag);
    
    if(!(flag & BP2318_MAV))
    {
     bp2318_err = 304;    // There is no measurement available
     return bp2318_err;
    }
*/

    if(bp2318_receive_scpi(instrID,"fetch?",buf) != 0)
      return bp2318_err;

    Scan(buf,"%s>%f",value);


	intRet = FindPattern (buf, 0, -1, "MOHM", 1, 0);

    if(FindPattern (buf, 0, -1, "MOHM", 1, 0) >= 0)
    	sprintf(unit,"%s","mOHM");
    else if (FindPattern (buf, 0, -1, "KOHM", 1, 0) >= 0)
    	sprintf(unit,"%s","kOHM");
    else if (FindPattern (buf, 0, -1, "OHM", 1, 0) >= 0)
    	sprintf(unit,"%s","OHM");
 	else
 	{

		//MT gendert am 01.08.2002
		//Das Gert sendet ber die Schnittstelle bei Messwerten <= 0.0
		//einen Messwert von 9.9999e+11
		//Dieser Fall wird hier korrigiert
		if (*value > 90000000000.0)
		{
			*value = 0.0;	  		// Messwert = '0'
			range = 3;		  		// Messbereich 'Ohm'
			CopyString(unit,0,func_unit[range],0,-1); 
			bp2318_err = 0;
			return bp2318_err;
		}	
	
	    if(bp2318_read_range(instrID,&range) != 0)
	    {
	      if(bp2318_err == 300)  /* NAK weil Software in 2318 kein*/
	      {
	        if(*value <= 0.02)   /* 20 mOhm*/
	         range = 1;
	        else if(*value <= 0.2)   /* 200 mOhm*/
	         range = 2;
	        else if(*value <= 2.0)   /* 2 Ohm*/
	         range = 3;
	        else if(*value <= 20.0)   /* 20Ohm*/
	         range = 4;
	        else if(*value <= 200.0)   /* 200 Ohm*/
	         range = 5;
	        else if(*value <= 2000.0)   /* 2KOhm*/
	         range = 6;
	        else if(*value <= 20000.0)   /* 20KOhm*/
	         range = 7;
	        else if(*value <= 200000.0)   /* 200KOhm*/
	         range = 8;

	        bp2318_err = 0;
	      }
	      else
	       return bp2318_err;     /* Messbereichslesen bei Z-Messart zult*/
	    }
	    CopyString(unit,0,func_unit[range],0,-1);

	    switch(range)
	    {
	     case 1: /* mOhm*/
	     case 2:
	      *value *= 1000.0;
	      break;
	     case 6:
	     case 7:
	     case 8: /* kOhm */
	      *value /= 1000.0;
	      break;
	    }
	}
    return bp2318_err;
}

/*=========================================================================*/
/* Function: Measure Stop                                                  */
/* Purpose:  This function stop the measring                               */
/*=========================================================================*/
int _VI_FUNC  bp2318_meas_stop(int instrID)
{
    bp2318_err = 0;
    if (bp2318_invalid_integer_range (instrID, 1, BP2318_MAX_INSTR -1, -1) != 0)
        return bp2318_err;

    if(bp2318_send_scpi(instrID,"ABORT") != 0)
      return bp2318_err;

    return bp2318_err;
}
/*=========================================================================*/
/* Function: Set Comparator Limit                                          */
/* Purpose:  This function set the comparator limit and state              */
/*=========================================================================*/
int _VI_FUNC  bp2318_set_comp_limit(int instrID,int state,double lower,double upper)
{
    bp2318_err = 0;
    if (bp2318_invalid_integer_range (instrID, 1, BP2318_MAX_INSTR -1, -1) != 0)
        return bp2318_err;
    if (bp2318_invalid_integer_range (state, 0, 1, -2) != 0)
        return bp2318_err;
    if (bp2318_invalid_real_range (lower, 0.0F, 200000.0F, -3) != 0)
        return bp2318_err;
    if (bp2318_invalid_real_range (upper, 0.0F, 200000.0F, -4) != 0)
        return bp2318_err;

    if(bp2318_send_scpi(instrID,func_comp_state[state]) != 0)
      return bp2318_err;

    Fmt(buf,"%s<CALC:LIM:LOW %f",lower);
    if(bp2318_send_scpi(instrID,buf) != 0)
      return bp2318_err;

    Fmt(buf,"%s<CALC:LIM:UPP %f",upper);
    if(bp2318_send_scpi(instrID,buf) != 0)
      return bp2318_err;

    return bp2318_err;
}
/*=========================================================================*/
/* Function: Get Comparator Limit                                          */
/* Purpose:  This function Get the comparator limit and state              */
/*=========================================================================*/
int _VI_FUNC  bp2318_get_comp_limit(int instrID,int* state,double* lower,double* upper)
{
    bp2318_err = 0;
    if (bp2318_invalid_integer_range (instrID, 1, BP2318_MAX_INSTR -1, -1) != 0)
        return bp2318_err;

    if(bp2318_receive_scpi(instrID,"CALC:LIM:STAT?",buf) != 0)
      return bp2318_err;

    Scan(buf,"%s>%d",state);

    if(bp2318_receive_scpi(instrID,"CALC:LIM:LOW?",buf) != 0)
      return bp2318_err;

    Scan(buf,"%s>%f",lower);

    if(bp2318_receive_scpi(instrID,"CALC:LIM:UPP?",buf) != 0)
      return bp2318_err;

    Scan(buf,"%s>%f",upper);

    return bp2318_err;
}
/*=========================================================================*/
/* Function: Set address                                                   */
/* Purpose:  This function set the address                                 */
/*=========================================================================*/
int _VI_FUNC  bp2318_set_address(int instrID,int g_addr,int u_addr)
{
    bp2318_err = 0;
    if (bp2318_invalid_integer_range (instrID, 1, BP2318_MAX_INSTR -1, -1) != 0)
        return bp2318_err;
    if (bp2318_invalid_integer_range (g_addr, 0, 15, -2) != 0)
        return bp2318_err;
    if (bp2318_invalid_integer_range (u_addr, 0, 15, -3) != 0)
        return bp2318_err;

    Fmt(buf,"%s<SYST:COMM:SER:ADDR:GRO %d",g_addr);
    if(bp2318_send_scpi(instrID,buf) != 0)
      return bp2318_err;

    Fmt(buf,"%s<SYST:COMM:SER:ADDR:USER %d",u_addr);
    if(bp2318_send_scpi(instrID,buf) != 0)
      return bp2318_err;

    return bp2318_err;
}
/*=========================================================================*/
/* Function: Get address                                                   */
/* Purpose:  This function get the address                                 */
/*=========================================================================*/
int _VI_FUNC  bp2318_get_address(int instrID,int* g_addr,int *u_addr)
{
    bp2318_err = 0;
    if (bp2318_invalid_integer_range (instrID, 1, BP2318_MAX_INSTR -1, -1) != 0)
        return bp2318_err;


    if(bp2318_receive_scpi(instrID,"SYST:COMM:SER:ADDR:GRO?",buf) != 0)
      return bp2318_err;

    Scan(buf,"%s>%d",g_addr);

    if(bp2318_receive_scpi(instrID,"SYST:COMM:SER:ADDR:USER?",buf) != 0)
      return bp2318_err;

    Scan(buf,"%s>%d",u_addr);


    return bp2318_err;
}
/*=========================================================================*/
/* Function: System Error                                                  */
/* Purpose:  This function Get the system error from the instrument        */
/*=========================================================================*/
int _VI_FUNC  bp2318_system_error(int instrID,char* err)
{
    bp2318_err = 0;
    if (bp2318_invalid_integer_range (instrID, 1, BP2318_MAX_INSTR -1, -1) != 0)
        return bp2318_err;

    if(bp2318_receive_scpi(instrID,"SYST:ERR?",err) != 0)
      return bp2318_err;

    return bp2318_err;
}

/*=========================================================================*/
/* Function: System Keyboard                                                  */
/* Purpose:  This function locked or unlocked the instrument keyboard      */
/*=========================================================================*/
int _VI_FUNC  bp2318_system_keyboard(int instrID,int lock)
{
    bp2318_err = 0;
    if (bp2318_invalid_integer_range (instrID, 1, BP2318_MAX_INSTR -1, -1) != 0)
        return bp2318_err;


    if(bp2318_send_scpi(instrID,func_keyboard[lock]) != 0)
      return bp2318_err;

    return bp2318_err;
}
/*=========================================================================*/
/* Function: Reset                                                         */
/* Purpose:  This function reset the instrument                            */
/*=========================================================================*/
int _VI_FUNC  bp2318_reset(int instrID)
{
    bp2318_err = 0;
    if (bp2318_invalid_integer_range (instrID, 1, BP2318_MAX_INSTR -1, -1) != 0)
        return bp2318_err;

    if(bp2318_send_scpi(instrID,"*RST") != 0)
      return bp2318_err;

    return bp2318_err;
}
/*=========================================================================*/
/* Function: Read ID                                                       */
/* Purpose:  This function read the ID from the instrument                 */
/*=========================================================================*/
int _VI_FUNC  bp2318_read_id(int instrID,char* id)
{
    bp2318_err = 0;
    if (bp2318_invalid_integer_range (instrID, 1, BP2318_MAX_INSTR -1, -1) != 0)
        return bp2318_err;

    if(bp2318_receive_scpi(instrID,"*IDN?",buf) != 0)
      return bp2318_err;

    Scan(buf,"%s>%s",id);

    return bp2318_err;
}
/*=========================================================================*/
/* Function: Display test                                                  */
/* Purpose:  This function activated display test                          */
/*=========================================================================*/
int _VI_FUNC  bp2318_display_test(int instrID)
{
    bp2318_err = 0;
    if (bp2318_invalid_integer_range (instrID, 1, BP2318_MAX_INSTR -1, -1) != 0)
        return bp2318_err;


    SetComTime (dev_port[instrID], 8.0);   /* You must wait for ACK 8.0 sec*/

    if(bp2318_send_scpi(instrID,"TEST:DISP") != 0)
      return bp2318_err;

    SetComTime (dev_port[instrID],std_timeout); /*Set the Standart Timeout*/

    return bp2318_err;
}

/*=========================================================================*/
/* This function SendSCPI                                                  */
/*=========================================================================*/
int _VI_FUNC  bp2318_send_scpi(int instrID,char* send_buffer)
{
   bp2318_err = 0;

   Fmt(buf,"%s<%s\n",send_buffer);
   bp2318_err = bp2318_send_x3_28(instrID,"sr",buf);
   if(bp2318_err != 0)
    return (bp2318_err);

   return(0);
}
/*=========================================================================*/
/* This function receive SCPI                                              */
/*=========================================================================*/
int _VI_FUNC  bp2318_receive_scpi(int instrID,char* send_buffer,char* receive_buffer)
{
   bp2318_err = 0;

   Fmt(buf,"%s<%s\n",send_buffer);
   bp2318_err = bp2318_send_x3_28(instrID,"sr",buf);
   if(bp2318_err != 0)
    return (bp2318_err);

   bp2318_err = bp2318_receive_x3_28(instrID,"po","",receive_buffer);
   if(bp2318_err != 0)
    return (bp2318_err);

   return(0);
}
/*=========================================================================*/
/*== UTILITY ROUTINES =====================================================*/
/*=========================================================================*/
int _VI_FUNC  bp2318_read_range (int instrID,int* range)
{
     bp2318_err = 0;
     if(bp2318_receive_scpi(instrID,":SENS:RES:RANG?",buf) != 0)
      return bp2318_err;

     if(CompareStrings (buf, 1, "20MOHM", 1, 1) == 0)
      *range = 1;
     else if(CompareStrings (buf, 1, "200MOHM", 1, 1) == 0)
      *range = 2;
     else if(CompareStrings (buf, 1, "2OHM", 1, 1) == 0)
      *range = 3;
     else if(CompareStrings (buf, 1, "20OHM", 1, 1) == 0)
      *range = 4;
     else if(CompareStrings (buf, 1, "200OHM", 1, 1) == 0)
      *range = 5;
     else if(CompareStrings (buf, 1, "2KOHM", 1, 1) == 0)
      *range = 6;
     else if(CompareStrings (buf, 1, "20KOHM", 1, 1) == 0)
      *range = 7;
     else if(CompareStrings (buf, 1, "200KOHM", 1, 1) == 0)
      *range = 8;
     else
      bp2318_err = 304;

     return bp2318_err;
}
int _VI_FUNC  bp2318_open_instr (int port,int baud_rate,int g_addr,int u_addr)
{
    int i, instrID;

    instrID = 0;

    /*--Check to see if the instrument is already in the Instrument Table.-*/

    for (i=1; i< BP2318_MAX_INSTR; i++)
    {
/*    FmtOut("\n g_adr %i u_adr %i i= %i",dev_group_addr[i],dev_user_addr[i],i);*/
     if((dev_group_addr[i] == g_addr) && (dev_user_addr[i] == u_addr))
     {
        instrID = i;
        i = BP2318_MAX_INSTR;
     }
    }

   /*------ If it is not in the instrument table, open an entry for the
            instrument. ---------------------------------------------------*/

    if (instrID <= 0)
    {
     for (i=1; i< BP2318_MAX_INSTR; i++)
     {
/*     FmtOut("\n g_adr %i u_adr %i i= %i",dev_group_addr[i],dev_user_addr[i],i);*/
        if( (dev_group_addr[i] < 0) && (dev_user_addr[i] < 0))
        {
         instrID = i;
         i = BP2318_MAX_INSTR;
        }
     }
    }

    /*----- If an entry could not be opened in the Instrument Table,
            return an error.-----------------------------------------------*/

    if (instrID <= 0)
    {
     bp2318_err = 220;
     return -1;
    }


    dev_port[instrID] = port;
    OpenComConfig (dev_port[instrID], "",baud_rate,0, 8, 1, 512, 512);
                                      
    if (rs232err != 0)
    {
    
     CloseCom (dev_port[instrID]);

     dev_port[instrID] = 0;
     dev_group_addr[instrID] = -1;
     dev_user_addr[instrID] = -1;
     bp2318_err = rs232err;
     return (-1);
    }
    dev_group_addr[instrID] = g_addr;
    dev_user_addr[instrID] = u_addr;
    return instrID;
}

/*=========================================================================*/
/* Function: Invalid Integer Range                                         */
/* Purpose:  This function checks an integer to see if it lies between a   */
/*           minimum and maximum value.  If the value is out of range, set */
/*           the global error variable to the value err_code.  If the      */
/*           value is OK, error = 0.                                       */
/*=========================================================================*/
int _VI_FUNC  bp2318_invalid_integer_range (int val,int min,int max,int err_code)
{
  if ((val < min) || (val > max)) {
    bp2318_err = err_code;
    return -1;
  }
  return 0;
}
/*=========================================================================*/
/* Function: Invalid Real Range                                            */
/* Purpose:  This function checks a real number to see if it lies between  */
/*           a minimum and maximum value.  If the value is out of range,   */
/*           set the global error variable to the value err_code.  If the  */
/*           value is OK, error = 0.                                       */
/*=========================================================================*/
int _VI_FUNC  bp2318_invalid_real_range (double val,double min,double max,int err_code)
{
  if ((val < min) || (val > max)) {
    bp2318_err = err_code;
    return -1;
  }
  return 0;
}
/*=========================================================================*/
/* Function: Close Instrument                                              */
/* Purpose:  This function closes the instrument by removing it from the   */
/*           GPIB device table and setting the address and the bd to zero  */
/*           in the Instrument Table.  The return value is equal to the    */
/*           global error variable.                                        */
/*=========================================================================*/
int _VI_FUNC  bp2318_close_instr (int instrID)
{
    if( (dev_group_addr[instrID] >= 0) && (dev_user_addr[instrID] >= 0))
    {
    
      CloseCom (dev_port[instrID]);
	  
      dev_port[instrID] = 0;
      dev_group_addr[instrID] = -1;
      dev_user_addr[instrID] = -1;
    }
    else
     bp2318_err = 221;

    return bp2318_err;
 }
/*=========================================================================*/
/* Function: Device Closed                                                 */
/* Purpose:  This function checks to see if the module has been            */
/*           initialized.  If the device has not been opened, a 1 is       */
/*           returned, 0 otherwise.                                        */
/*=========================================================================*/
int _VI_FUNC  bp2318_device_closed (int instrID)
{
     if( (dev_group_addr[instrID] < 0) && (dev_user_addr[instrID] < 0))
     {
        bp2318_err = 232;
        return -1;
     }
     return 0;
}

/*=========================================================================*/
/* This function send a comand and message and wait for ACK                */
/*=========================================================================*/
int _VI_FUNC  bp2318_send_x3_28(int instrID,char* cmd,char* msg)
{
    int ret,r_byte, repeat;

    FlushInQ(dev_port[instrID]);           /* Clear Port */

    bcc_repeat = 0;
    for(repeat = 0;repeat < 100;++repeat)
    {
     ret = bp2318_send_x3_28_message(instrID,cmd,msg);
     if(ret != 0)
       return (ret);     /* ERROR*/

     r_byte = ComRdByte(dev_port[instrID]);
     if (rs232err != 0)
      return (231);

   /*  FmtOut("\nsend ret %c %d\n",r_byte,bcc_repeat);*/
     switch(r_byte)
     {
      case ACK:
       return  (0);
       break;
      case NAK:
       if(bcc_repeat >= max_bcc_repeat)
        return (300);                /* NAK => MAX Repeat */
       bcc_repeat++;
       break;
      case EOT:
       return (301);
       break;
     }
    }

    return (0);
}

/*=========================================================================*/
/* This function send comand and message and receive a string              */
/*=========================================================================*/
int _VI_FUNC  bp2318_receive_x3_28(int instrID,char* cmd,char* msg,char* receive_buf)
{
    int ret;

    FlushInQ(dev_port[instrID]);           /* Clear Port */
    ret = bp2318_send_x3_28_message(instrID,cmd,msg);
    if(ret != 0)
      return (ret);     /* ERROR*/

    ret = bp2318_receive_and_ack(instrID,receive_buf);
    if(ret != 0)
      return (ret);     /* ERROR*/

    ComRdByte(dev_port[instrID]);   /* Read the EOT from Port*/


	// MT Test 24.11.2006
 //   ComSetEscape (dev_port[instrID], CLRRTS);								// Clear RTS 


    return(0);
}
/*=========================================================================*/
/* This function receive a string                                          */
/*=========================================================================*/
int _VI_FUNC  bp2318_receive_and_ack(int instrID,char* receive_buf)
{
    int r_byte,r_bcc,bcc,l,first_byte,i;

/*    l = StringLength(receive_buf);
    for( i = 0; i < l;i++)
     receive_buf[i] = 0;*/

    bcc_repeat = 0;
    for(;;)
    {
     first_byte = ComRdByte(dev_port[instrID]);   /* Read the EOT or STX from Port*/
     if(first_byte == EOT)
      return (301);


     r_byte = ComRdTerm(dev_port[instrID],in_buf,MAX_BYTE-1,ETX);/* Read to ETX*/
     if (rs232err != 0)
      return (231);

     if(bcc_state == 1)
     {
      r_bcc = ComRdByte(dev_port[instrID]);   /* Read the BCC from Port*/
      if (rs232err != 0)
       return (231);
     }
     in_buf[r_byte] = ETX;            /* Calculate bcc with ETX */
     in_buf[r_byte+1] = NUL;          /* Terminate the String*/
     l = StringLength(in_buf);

    /* FmtOut("\nin %s",in_buf);*/

     CopyString (swap_buf, 0,in_buf, 0, l);   /* calculate bcc without STX */
     if(bcc_state == 1)
     {
      bcc = bp2318_calculate_bcc(swap_buf);
      if(r_bcc != bcc)
      {
       ComWrtByte(dev_port[instrID],NAK);
       if(bcc_repeat >= max_bcc_repeat)
        return (300);                /* NAK => MAX Repeat */
       bcc_repeat++;
      }
      else
      {
       ComWrtByte(dev_port[instrID],ACK);
       CopyString (receive_buf, 0,in_buf,0,l-3); /* Copy without LF,ETX*/
       return(0);                /* 0 = ACK == OK*/
      }
     }
     else    /* Without bcc check*/
     {
      ComWrtByte(dev_port[instrID],ACK);
      CopyString (receive_buf, 0,in_buf,0,l-3); /* Copy without LF,ETX*/
      return(0);                /* 0 = ACK == OK*/
     }
    }

    return(0);
}

/*=========================================================================*/
/* This function send a comand and message                                 */
/*=========================================================================*/
int _VI_FUNC  bp2318_send_x3_28_message(int instrID,char* cmd,char* msg)
{
    int ret,l;
    int bcc;

    ret = CompareStrings (msg, 0,"", 0, 0); /* Send a command without msg ?*/
    if(ret == 0)
     Fmt(out_buf,"%s<%c%x%x%x%x%s%c", EOT,
                                      dev_group_addr[instrID],
                                      dev_group_addr[instrID],
                                      dev_user_addr[instrID],
                                      dev_user_addr[instrID],
                                      cmd,ENQ); /* without msg*/
    else
    {
     Fmt(swap_buf ,"%s<%s%c",msg,ETX);   /* with msg*/
     if(bcc_state == 1)
     {
      bcc = bp2318_calculate_bcc(swap_buf);
      Fmt(out_buf,"%s<%c%x%x%x%x%s%c%s%c", EOT,
                                           dev_group_addr[instrID],
                                           dev_group_addr[instrID],
                                           dev_user_addr[instrID],
                                           dev_user_addr[instrID],
                                           cmd,STX,swap_buf,bcc);
     }
     else
      Fmt(out_buf,"%s<%c%x%x%x%x%s%c%s", EOT,
                                         dev_group_addr[instrID],
                                         dev_group_addr[instrID],
                                         dev_user_addr[instrID],
                                         dev_user_addr[instrID],
                                         cmd,STX,swap_buf);
    }

  /* FmtOut("\nout %s",out_buf);*/

    l = StringLength(out_buf);
    
    ret = ComWrt(dev_port[instrID], out_buf, l);     /* Write the Message */
    if (rs232err != 0)
     return (230);
     

    return (0);
}
/*=========================================================================*/
/* This function calculate the block check                                 */
/*=========================================================================*/
int _VI_FUNC  bp2318_calculate_bcc(char* msg)
{
    int l,i,bcc;

    bcc = 0;
    l = StringLength(msg);
    for(i = 0;i <= l;i++)
     bcc ^= msg[i];     /* exclusive OR */

    bcc |= 0x80;    /* OR Bit 8 */

    return (bcc);
}
/*=========================================================================*/
/* This function setup the function arrays                                 */
/*=========================================================================*/
void bp2318_setup_arrays (void)
{
    int i;

    if(!setup_first)/*----------- Reset the Arrays before initial --------*/
    {
     for (i=1; i< BP2318_MAX_INSTR; i++)
     {
       dev_group_addr[i] = -1;
       dev_user_addr[i]  = -1;
       dev_port[i]       =  0;
     }
     setup_first = 1;
    }

    int_level[0] = 4;
    int_level[1] = 3;
    int_level[2] = 4;
    int_level[3] = 3;

    func_load[0] = ":SENS:RES:LOAD REAL";
    func_load[1] = ":SENS:RES:LOAD COMP";

    func_auto_zero[0] = ":CAL:ZERO OFF";
    func_auto_zero[1] = ":CAL:ZERO ON";

    func_range[0] = ":SENS:RES:RANG:AUTO ON";
    func_range[1] = ":SENS:RES:RANG 10MOHM";
    func_range[2] = ":SENS:RES:RANG 100MOHM";
    func_range[3] = ":SENS:RES:RANG 1OHM";
    func_range[4] = ":SENS:RES:RANG 10OHM";
    func_range[5] = ":SENS:RES:RANG 100OHM";
    func_range[6] = ":SENS:RES:RANG 1KOHM";
    func_range[7] = ":SENS:RES:RANG 10KOHM";
    func_range[8] = ":SENS:RES:RANG 100KOHM";

    func_unit[0] = "NONE";
    func_unit[1] = "mOHM";
    func_unit[2] = "mOHM";
    func_unit[3] = "OHM";
    func_unit[4] = "OHM";
    func_unit[5] = "OHM";
    func_unit[6] = "kOHM";
    func_unit[7] = "kOHM";
    func_unit[8] = "kOHM";

    func_limit_20mV[0] = ":SOUR:VOLT:LIM NONE";
    func_limit_20mV[1] = ":SOUR:VOLT:LIM 20MV,1A";
    func_limit_20mV[2] = ":SOUR:VOLT:LIM 20MV,100MA";
    func_limit_20mV[3] = ":SOUR:VOLT:LIM 20MV,10MA";

    func_temp_comp[0] = ":SENS:CORR:TCOM:MED NONE";
    func_temp_comp[1] = ":SENS:CORR:TCOM:MED CU/AL";
    func_temp_comp[2] = ":SENS:CORR:TCOM:MED MS63";
    func_temp_comp[3] = ":SENS:CORR:TCOM:MED MS80";

    func_comp_state[0] = ":CALC:LIM:STAT OFF";
    func_comp_state[1] = ":CALC:LIM:STAT ON";

    func_keyboard[0] = "SYST:KLOC OFF";
    func_keyboard[1] = "SYST:KLOC ON";

}
/*= THE END ===============================================================*/

